<?php

namespace App\Sys\Services\Hajj;

use App\Models\Countries;
use App\Models\CountryVisa;
use App\Models\hajjPayment;
use App\PaymentAPI;
use App\Sys\Repository\hajjPaymentRepo;
use App\Sys\Repository\hajjRepo;
use App\Sys\Services;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\URL;
use Illuminate\Support\Str;

class HajjServices extends Services
{
    use PaymentAPI;

    private $hajj;
    private $payment;

    public function __construct()
    {
        $this->hajj = new hajjRepo();
        $this->payment = new hajjPaymentRepo();
    }

    public function add($data)
    {
        // Validate the data

        $rules = [
            'national' => 'required|integer|exists:mysql2.countries,id',
            'country_id' => $data['type'] == 'visa'
                ? 'required|exists:mysql2.countries,id'
                : 'sometimes|nullable',
            'country_code' => 'required',
            'type' => 'required|in:hajj,umrah,visa',
            'city_id' => 'nullable|integer|exists:mysql2.cities,id',
            'phone' => 'required|numeric',
            'email' => 'nullable|email',
            'detials' => 'required|array|min:1',
            'detials.*.first_name_en' => 'required|string|max:500',
            'detials.*.first_name_ar' => 'nullable|string|max:500',
            'detials.*.last_name_en' => 'required|string|max:500',
            'detials.*.last_name_ar' => 'nullable|string|max:500',
            'detials.*.second_name_en' => 'required|string|max:500',
            'detials.*.second_name_ar' => 'nullable|string|max:500',
            'detials.*.third_name_en' => 'required|string|max:500',
            'detials.*.third_name_ar' => 'nullable|string|max:500',
            'detials.*.birth_day' => 'required|integer|min:1|max:31',
            'detials.*.birth_month' => 'required|integer|min:1|max:12',
            'detials.*.birth_year' => 'required|integer|min:1900|max:' . date('Y'),
            'detials.*.passport_number' => 'required',
            'detials.*.passport_doc' => 'required|active_url',
            'detials.*.passport_expire_day' => 'required|integer|min:1|max:31',
            'detials.*.passport_expire_month' => 'required|integer|min:1|max:12',
            'detials.*.passport_expire_year' => 'required|integer|min:' . date('Y'),
            'detials.*.photo' => ['required', 'active_url'],
            'detials.*.hosting_file' => ['nullable', 'active_url'],


        ];
        $validator = validator($data, $rules);
        $countrty = Countries::find($data['national']);
        $validator->after(function ($validator) use ($data, $countrty) {
            foreach ($data['detials'] as $index => $item) {
                if ($countrty && $countrty->id == 24 && $data['type'] == "umrah" && $index == 0) {
                    if ($item['hosting_file'] == null) {
                        $validator->errors()->add("detials.$index.hosting_file", 'The hosting file is required for the first person when country is Sudan.');
                    }
                }
            }

        });
        if ($data['type'] == "visa") {
            $country_visa = CountryVisa::where([
                ['country_id', $data['national']], ['destination_id', $data['country_id']]
            ])->first();
            if (empty($country_visa)) {
                $validator->after(function ($validator) use ($data) {
                    $validator->errors()->add("country_id", 'visa to country not a valid');
                });
            }
        }
        if ($validator->fails()) {
            $this->setError($validator->errors());
            return false;
        }
        $data['price_child'] = $data['type'] == "visa" ? $country_visa->price_adult : $countrty->{$data['type'] . '_price_adult'};
        $data['price_adult'] = $data['type'] == "visa" ? $country_visa->price_child : $countrty->{$data['type'] . '_price_child'};

        return $this->hajj->add($data);

    }

    public
    function saveFiles($data)
    {
        $rules = [
            'file_type' => 'required|in:photo,passport_doc,hosting_file',
            'file' => 'required|string',
        ];
        $validator = validator($data, $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());
            return false;
        }

        $photoFolder = 'uploads/photos/';
        $fileFolder = 'uploads/files/';
        if ($data['file_type'] == "passport_doc") {
            $passportDir = public_path($fileFolder);
            if (!file_exists($passportDir)) {
                mkdir($passportDir, 0777, true);
            }

            if (
                preg_match('/^data:(image\/[\w\+\-\.]+|application\/pdf);base64,/', $data['file'], $type)
            ) {
                $mimeType = $type[1];

                // قائمة أنواع الملفات المدعومة
                $mimeToExt = [
                    'image/jpeg' => 'jpg',
                    'image/jpg' => 'jpg',
                    'image/png' => 'png',
                    'image/gif' => 'gif',
                    'image/webp' => 'webp',
                    'image/bmp' => 'bmp',
                    'image/tiff' => 'tiff',
                    'image/svg+xml' => 'svg',
                    // 'application/pdf' => 'pdf',
                ];

                if (!array_key_exists($mimeType, $mimeToExt)) {
                    $this->setError(["file" => ['Only image files (JPG, PNG, GIF, BMP, WEBP, TIFF, SVG)  are allowed.']]);
                    return false;
                }

                $fileExt = $mimeToExt[$mimeType];

                // إزالة base64 header
                $base64Data = preg_replace('/^data:(image\/[\w\+\-\.]+|application\/pdf);base64,/', '', $data['file']);
                $fileData = base64_decode($base64Data);

                if ($fileData === false) {
                    $this->setError(["file" => ['Invalid base64 file data.']]);
                    return false;
                }

                $fileName = uniqid($data['file_type'] . '_') . Str::random(10) . '.' . $fileExt;
                $filePath = public_path($fileFolder . $fileName);
                file_put_contents($filePath, $fileData);

                return url($fileFolder . $fileName);

            } else {
                $this->setError(["file" => ['The passport document must be a valid image or PDF file (base64 encoded).']]);
                return false;
            }
        }
        // PHOTO
        if ($data['file_type'] == "photo") {
            $photoDir = public_path($photoFolder);
            if (!file_exists($photoDir)) {
                mkdir($photoDir, 0777, true); // إنشاء المجلدات بشكل متداخل
            }
            if (preg_match('/^data:image\/(\w+);base64,/', $data['file'], $type)) {
                $photoExt = $type[1];
                $photoData = base64_decode(preg_replace('/^data:image\/\w+;base64,/', '', $data['file']));

                $photoName = uniqid('photo_') . Str::random(10) . '.' . $photoExt;
                $photoPath = public_path($photoFolder . $photoName);
                file_put_contents($photoPath, $photoData);

                return url($photoFolder . $photoName); // <== Full URL
            } else {
                $this->setError(["file" => ['The file must be a valid image']]);
                return false;
            }
        }
        // hosting_file
        if ($data['file_type'] == 'hosting_file') {
            $fileDir = public_path($fileFolder);
            if (!file_exists($fileDir)) {
                mkdir($fileDir, 0777, true);
            }

            if (preg_match('/^data:application\/([\w\.\-]+);base64,/', $data['file'], $type)) {
                $mimeType = $type[1];
                $mimeFull = 'application/' . $mimeType;
                $mimeToExt = [
                    'pdf' => 'pdf',
                    'vnd.openxmlformats-officedocument.wordprocessingml.document' => 'docx',
                    'msword' => 'doc',
                    'vnd.ms-excel' => 'xls',
                    'vnd.openxmlformats-officedocument.spreadsheetml.sheet' => 'xlsx'
                ];
                if ($mimeType !== 'pdf') {
                    $this->setError(['file' => ['Only pdf files are allowed.']]);
                    return false;
                }
                $fileExt = $mimeToExt[$mimeType] ?? 'pdf';
                $base64Data = preg_replace('/^data:application\/[\w\.\-]+;base64,/', '', $data['file']);
                $fileData = base64_decode($base64Data);

                if ($fileData !== false) {
                    $fileName = uniqid('file_') . Str::random(10) . '.' . $fileExt;
                    $filePath = public_path($fileFolder . $fileName);
                    file_put_contents($filePath, $fileData);

                    return url($fileFolder . $fileName); // الرابط الكامل
                } else {

                    $this->setError(['file' => ['Not a valid base64 encoded file.']]);
                    return false;
                }
            } else {

                $this->setError(['file' => ['Not a valid base64 encoded file.']]);
                return false;
            }
        }

        $this->setError(['file' => ['Not Cant save any files.']]);
        return false;
    }

    public
    function list()
    {
        return $this->hajj->list();
    }

    public function getDeitals($id)
    {
        return $this->hajj->getDeitals($id);
    }

    public function payment($id)
    {
        // get One
        $hajj = $this->hajj->getByid($id);
        if (
            $hajj->user_id == Auth()->guard('api')->user()->id &&
            in_array($hajj->status, ["waiting_for_payment", "payment_errors"]) &&
            $hajj->date_exp >= Carbon::now()
        ) {
            $data = [
                'hajj_id' => $id,
                'currency_code' => 'AED', // AED
                "value" => intval($hajj->total * 100)
            ];
            $createdPayment = $this->payment->add($data);
            $url = route('hajj_payment-webview', ['payment_code' => $createdPayment->id, 'user_id' => $createdPayment->user_id]);
            $checkStatus = route('hajj.checkingBooking', ['id' => $createdPayment->id]);
            return [
                "url" => $url,
                "payment_code" => $createdPayment->id,
                'check_status_url' => $checkStatus,
                'pramater_one' => $createdPayment->id,
                'pramater_two' => $createdPayment->user_id,
            ];

        }
        $this->setError('This request cannot be processed. It may have expired, already been paid, or does not belong to the current user.');
        return false;
    }

    public function checkPayment($id)
    {
        return $this->payment->getByid($id);
    }

    public function pay($request)
    {
        // get content
        $content = $request->getContent();
        $json = json_decode($content);
        $payment_code = $json->payment_code;
        $user = $json->the_code;
        $redirect_url = URL::route('finished');
        $payment = $this->payment->getByid($payment_code);
        if (!empty($payment)) {
            if ($payment->expires_at < now()) {
                $this->setError([
                    'status' => 419,
                    'message' => 'pay',
                    'error' => 'session expired'
                ]);
                return false;
            }
            if ($payment->user_id != $user) {
                $this->setError([
                    'status' => 403,
                    'message' => 'pay',
                    'error' => 'access denied'
                ]);
                return false;
            }

            $params = [
                "action" => $payment->action ?? "PURCHASE",
                "amount" => [
                    "currencyCode" => $payment->currency_code ?? "AED",
                    "value" => $payment->value
                ],
                "language" => "en",
                "payment" => [
                    "currency" => $payment->currency_code ?? "AED"
                ],
                "emailAddress" => $payment->getUser?->client_email ?? 'msayed0@outlook.com',
                "merchantAttributes" => [
                    "redirectUrl" => $redirect_url,
                    "cancelUrl" => $redirect_url,
                    "skip3DS" => false,
                    "skipConfirmationPage" => false
                ]
            ];

            try {
                $response = $this->PaymentAPI(
                    $params,
                    [
                        'Accept' => "application/vnd.ni-payment.v2+json",
                        'Content-type' => "application/vnd.ni-payment.v2+json",
                        'Authorization' => "Bearer " . $payment->payment_token,
                    ],
                    "transactions/outlets/" . config('services.third_party.outlet') . "/payment/hosted-session/" . $json->sessionID
                    , "post"
                );

                $pa = hajjPayment::find($payment_code);
                $pa->session_id = $json->sessionID;
                $pa->token = $json->token;
                $pa->pay_response = json_encode($response);
                $pa->save();

                return $response;
            } catch (\Throwable $th) {
                $this->setError([
                    'status' => $th->getCode() == 0 ? 500 : $th->getCode(),
                    'message' => 'pay',
                    'error' => $th->getMessage()
                ]);
                return false;
            }
        }
        $this->setError([
            'status' => 0,
            'message' => 'pay',
            'error' => 'not found payment'
        ]);
        return false;

    }

    public function getAuthToken($request)
    {
        $content = $request->getContent();
        $json = json_decode($content);
        $payment_code = $json->payment_code;
        $response = $this->PaymentAPI(
            ['grant_type' => 'client_credentials'],
            [
                "Content-Type" => "application/vnd.ni-identity.v1+json",
                "Authorization" => "Basic " . config("services.third_party.webhook")
            ], "identity/auth/access-token", "post"
        );

        if (isset($response['access_token']) && $response['access_token'] != null) {
            if ($generate = $this->payment->update($response, $payment_code, $response)) {
                return $generate;
            }
        }
        $this->setError([
            'status' => 400,
            'message' => 'access token',
            'errors' => 'order not found try again'
        ]);
        return false;

    }

    public function PaymentStatus($data)
    {
        $payment = $this->payment->getByid($data['payment_code']);
        if (!empty($payment)) {
            $orders = json_decode($payment->pay_response, true);
            $respo = $this->checkStatus($orders);
            $status = isset($respo['_embedded']['payment'][0]['state']) ? $respo['_embedded']['payment'][0]['state'] : $data['payment_status'];
            $payment->payment_status = $status;
            // $payment->status = $status == "CAPTURED" ? "paid" : "payment_errors";
            $payment->save();
            if ($status == "CAPTURED") {
                $this->hajj->updateStatus($payment->hajj_request_id, 'paid');

                return ['status' => 200,
                    'message' => 'check status',
                    'payment_status' => $status];
            } else {
                $this->hajj->updateStatus($payment->hajj_request_id, 'waiting_for_payment');
                $this->setError(['status' => 400,
                    'message' => 'check status',
                    'payment_status' => $status]);
                return false;
            }
        }

        $this->setError(['status' => 400,
            'message' => 'check status',
            'payment_status' => 'cansel']);
        return false;

    }


    public
    function checkStatus($orders)
    {
        $token = $this->getAccsessToken();
        $end = 'transactions/outlets/' . $orders['outletId'] . '/orders/' . $orders['orderReference'];
        $response = $this->PaymentAPI(null,
            [
                'Accept' => "application/vnd.ni-payment.v2+json",
                'Content-type' => "application/vnd.ni-payment.v2+json",
                'Authorization' => "Bearer " . $token,
            ],
            $end
            , "GET"
        );

        return $response;
    }

    public
    function getAccsessToken()
    {
        $response = $this->PaymentAPI(
            ['grant_type' => 'client_credentials'],
            [
                "Content-Type" => "application/vnd.ni-identity.v1+json",
                "Authorization" => "Basic " . config("services.third_party.webhook")
            ], "identity/auth/access-token", "post"
        );

        return $response['access_token'];
    }

    public function checkingBooking($id)
    {
        $payment = $this->payment->getByid($id);
        if (!empty($payment)) {
            return [
                'payment_status' => $payment->payment_status,
                'status' => $payment->getHajj?->status,
                'last_updated' => $payment->updated_at
            ];
        }
        return false;
    }

    public function getCountryPrices($data)
    {
        $rules = [
            'country_id' => 'required|integer|exists:mysql2.countries,id',
            'destination_id' => $data['type'] == 'visa'
                ? 'required|exists:mysql2.countries,id'
                : 'sometimes|nullable',
            'type' => 'required|in:hajj,umrah,visa',
        ];
        $validator = validator($data, $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());
            return false;
        }
        if ($data['type'] == 'visa') {
            $country = CountryVisa::where([
                ['country_id', $data['country_id']], ['destination_id', $data['destination_id']]
            ])->first();
            if (!empty($country)) {
                return [
                    "price_adult" => $country->price_adult,
                    "price_child" => $country->price_child,
                    "need_hosting_file" => false
                ];
            }
            $this->setError('no visa available');
            return false;
        }
        $colmue = 'accepts_' . $data['type'] . '_visa';

        $country = Countries::where([
            [$colmue, 1], ['id', $data['country_id']]
        ])->first();
        if (!empty($country)) {
            $adult = $data['type'] . "_price_adult";
            $child = $data['type'] . "_price_child";
            return [
                "price_adult" => $country->{$adult},
                "price_child" => $country->{$child},
                "need_hosting_file" => $country->id == 24 && $data['type'] == "umrah" ? true : false,
            ];
        }

        $this->setError('no visa available');
        return false;

    }


}

/**
 * $photoFolder = 'uploads/photos/';
 * $fileFolder = 'uploads/files/';
 * foreach ($data['detials'] as $index => $item) {
 * // PASSPORT DOC
 * if (!empty($item['passport_doc'])) {
 * $passportDir = public_path($fileFolder);
 * if (!file_exists($passportDir)) {
 * mkdir($passportDir, 0777, true);
 * }
 *
 * if (
 * preg_match('/^data:(image\/[\w\+\-\.]+|application\/pdf);base64,/', $item['passport_doc'], $type)
 * ) {
 * $mimeType = $type[1];
 *
 * // قائمة أنواع الملفات المدعومة
 * $mimeToExt = [
 * 'image/jpeg'      => 'jpg',
 * 'image/jpg'       => 'jpg',
 * 'image/png'       => 'png',
 * 'image/gif'       => 'gif',
 * 'image/webp'      => 'webp',
 * 'image/bmp'       => 'bmp',
 * 'image/tiff'      => 'tiff',
 * 'image/svg+xml'   => 'svg',
 * // 'application/pdf' => 'pdf',
 * ];
 *
 * if (!array_key_exists($mimeType, $mimeToExt)) {
 * $this->setError(["detials.$index.passport_doc" => ['Only image files (JPG, PNG, GIF, BMP, WEBP, TIFF, SVG)  are allowed.']]);
 * return false;
 * }
 *
 * $fileExt = $mimeToExt[$mimeType];
 *
 * // إزالة base64 header
 * $base64Data = preg_replace('/^data:(image\/[\w\+\-\.]+|application\/pdf);base64,/', '', $item['passport_doc']);
 * $fileData = base64_decode($base64Data);
 *
 * if ($fileData === false) {
 * $this->setError(["detials.$index.passport_doc" => ['Invalid base64 file data.']]);
 * return false;
 * }
 *
 * $fileName = uniqid('passport_') . Str::random(10) . '.' . $fileExt;
 * $filePath = public_path($fileFolder . $fileName);
 * file_put_contents($filePath, $fileData);
 *
 * $data['detials'][$index]['passport_doc_link'] = url($fileFolder . $fileName);
 *
 * } else {
 * $this->setError(["detials.$index.passport_doc" => ['The passport document must be a valid image or PDF file (base64 encoded).']]);
 * return false;
 * }
 * }
 * // PHOTO
 * if ($item['photo'] != null) {
 * $photoDir = public_path($photoFolder);
 * if (!file_exists($photoDir)) {
 * mkdir($photoDir, 0777, true); // إنشاء المجلدات بشكل متداخل
 * }
 * if(preg_match('/^data:image\/(\w+);base64,/', $item['photo'], $type))
 * {
 * $photoExt = $type[1];
 * $photoData = base64_decode(preg_replace('/^data:image\/\w+;base64,/', '', $item['photo']));
 *
 * $photoName = uniqid('photo_') . Str::random(10) . '.' . $photoExt;
 * $photoPath = public_path($photoFolder . $photoName);
 * file_put_contents($photoPath, $photoData);
 *
 * $data['detials'][$index]['photo_link'] = url($photoFolder . $photoName); // <== Full URL
 * }
 *
 * }
 * if($data['national'] == 24 && $index == 0 && $data['type'] =="umrah")
 * {
 * if ($item['hosting_file'] != null) {
 * $fileDir = public_path($fileFolder);
 * if (!file_exists($fileDir)) {
 * mkdir($fileDir, 0777, true);
 * }
 *
 * if (preg_match('/^data:application\/([\w\.\-]+);base64,/', $item['hosting_file'], $type)) {
 * $mimeType = $type[1];
 * $mimeFull = 'application/' . $mimeType;
 * $mimeToExt = [
 * 'pdf' => 'pdf',
 * 'vnd.openxmlformats-officedocument.wordprocessingml.document' => 'docx',
 * 'msword' => 'doc',
 * 'vnd.ms-excel' => 'xls',
 * 'vnd.openxmlformats-officedocument.spreadsheetml.sheet' => 'xlsx'
 * ];
 * if ($mimeType !== 'pdf') {
 * $this->setError(['detials.0.hosting_file' => ['Only pdf files are allowed.']]);
 * return false;
 * }
 * $fileExt = $mimeToExt[$mimeType] ?? 'pdf';
 * $base64Data = preg_replace('/^data:application\/[\w\.\-]+;base64,/', '', $item['hosting_file']);
 * $fileData = base64_decode($base64Data);
 *
 * if ($fileData !== false) {
 * $fileName = uniqid('file_') . Str::random(10) . '.' . $fileExt;
 * $filePath = public_path($fileFolder . $fileName);
 * file_put_contents($filePath, $fileData);
 *
 * $data['detials'][$index]['hosting_link'] = url($fileFolder . $fileName); // الرابط الكامل
 * } else {
 *
 * $data['detials'][$index]['hosting_link'] = null;
 * }
 * } else {
 *
 * $this->setError(['detials.0.hosting_file'=>['Not a valid base64 encoded file.']]);
 * return false;
 * }
 * }else{
 * $this->setError(['detials.0.hosting_file'=>['The hosting file is required for person when country is Sudan.']]);
 * return false;
 * }
 * }
 * }
 */
